let STATE = {
    enabled: true,
    profiles: [],
    whitelist: [],
    blacklist: [],
    pauseUntil: null,
    hideComments: true,
    proximityLimit: 12,

    // IA locale
    aiEnabled: true,
    aiMemory: {},       // { mot: score }
    aiMinHits: 3        // seuil apprentissage
};

const SCORE_LIMIT = 6;

const HOST = location.hostname;

// heuristique légère pour éviter de flinguer les commentaires (option)
const COMMENT_SELECTORS = [
    "comment",
    "comments",
    "reply",
    "replies",
    "responses",
    "ytd-comment-thread-renderer",
    "ytd-comments"
];

chrome.storage.sync.get(null, (data) => {
    Object.assign(STATE, data);
    if (isActive()) init();
});

chrome.storage.onChanged.addListener((changes) => {
    for (const k in changes) STATE[k] = changes[k].newValue;
});

function isTitle(el) {
    return ["H1", "H2", "H3", "H4"].includes(el.tagName);
}

// Calcule la distance minimale (en nombre de mots)
// entre un alias et un mot spoiler dans un texte
function wordsDistance(text, aliasWords, spoilerWords) {
    const words = text.split(/\s+/);

    let aliasPos = [];
    let spoilerPos = [];

    words.forEach((w, i) => {
        if (aliasWords.includes(w)) aliasPos.push(i);
        if (spoilerWords.includes(w)) spoilerPos.push(i);
    });

    let min = Infinity;

    aliasPos.forEach(i => {
        spoilerPos.forEach(j => {
            min = Math.min(min, Math.abs(i - j));
        });
    });

    return min;
}

function scanYouTube() {
    document.querySelectorAll(
        "ytd-video-renderer #video-title, ytd-rich-item-renderer #video-title"
    ).forEach(el => {
        if (!el.textContent) return;
        if (shouldMask(el.textContent)) mask(el);
    });
}

function scanReddit() {
    document.querySelectorAll("h3").forEach(el => {
        if (!el.textContent) return;
        if (shouldMask(el.textContent)) mask(el);
    });
}

// Decide si on doit agir sur cette page
function isActive() {
    if (!STATE.enabled) return false;
    if (STATE.pauseUntil && Date.now() < STATE.pauseUntil) return false;

    const host = location.hostname;
    if (Array.isArray(STATE.blacklist) && STATE.blacklist.includes(host)) return false;

    if (Array.isArray(STATE.whitelist) && STATE.whitelist.length > 0) {
        if (!STATE.whitelist.includes(host)) return false;
    }
    return true;
}

function computeScore(text, profile) {
    const t = text.toLowerCase();
    let score = 0;

    (profile.keywords?.fort || []).forEach(w => { if (t.includes(w)) score += 5; });
    (profile.keywords?.moyen || []).forEach(w => { if (t.includes(w)) score += 3; });
    (profile.keywords?.faible || []).forEach(w => { if (t.includes(w)) score += 1; });

    // 🧠 IA locale (pondération adaptative)
    if (STATE.aiEnabled && STATE.aiMemory) {
        Object.entries(STATE.aiMemory).forEach(([word, hits]) => {
            if (hits >= STATE.aiMinHits && t.includes(word)) {
                score += Math.min(4, Math.floor(hits / 2));
            }
        });
    }

    return score;
}

function learnFrom(text) {
    if (!STATE.aiEnabled) return;

    const words = text
        .toLowerCase()
        .replace(/[^\w\s]/g, "")
        .split(/\s+/)
        .filter(w => w.length > 4);

    words.forEach(w => {
        STATE.aiMemory[w] = (STATE.aiMemory[w] || 0) + 1;
    });

    chrome.storage.local.set({ aiMemory: STATE.aiMemory });
}

function hasAlias(text, profile) {
    const t = text.toLowerCase();
    return (profile.aliases || []).some(a => a && t.includes(a.toLowerCase()));
}

function shouldMask(text) {
    for (const profile of (STATE.profiles || [])) {
        if (!profile?.enabled) continue;
        if (!hasAlias(text, profile)) continue;

        const score = computeScore(text, profile);
        if (score >= SCORE_LIMIT) {

            // si pas de limite définie → comportement normal
            if (!STATE.proximityLimit) return true;

            const aliasWords = profile.aliases
                .flatMap(a => a.toLowerCase().split(/\s+/));

            const spoilerWords = [
                ...profile.keywords.fort,
                ...profile.keywords.moyen,
                ...profile.keywords.faible
            ].map(w => w.toLowerCase());

            const distance = wordsDistance(
                text.toLowerCase(),
                aliasWords,
                spoilerWords
            );
            console.log("[AntiSpoil] Profil :", profile.name, "Score :", score);
            if (distance <= STATE.proximityLimit) return true;
        }

    }
    return false;
}

function isLikelyCommentNode(el) {
    const cls = (el.className || "").toString().toLowerCase();
    if (COMMENT_SELECTORS.some(s => cls.includes(s))) return true;
    // fallback: si un parent contient "comment"
    const parent = el.closest('[class*="comment"], [id*="comment"]');
    return !!parent;
}

function mask(el) {
    chrome.storage.sync.get(["stats"], d => {
        const stats = d.stats || {};
        stats.masks = (stats.masks || 0) + 1;
        chrome.storage.sync.set({ stats });
    });
    if (el.dataset.antispoil) return;
    el.dataset.antispoil = "1";

    el.classList.add("anti-spoil-mask");

    // badge discret (non intrusif)
    const badge = document.createElement("span");
    badge.className = "anti-spoil-badge";
    badge.textContent = "spoiler";
    badge.style.filter = "none"; // ne pas flouter le badge
    el.appendChild(badge);

    el.addEventListener("click", () => {
        el.classList.add("revealed");
        badge.remove();
    }, { once: true });
    // 🧠 apprentissage uniquement sur contenu masqué
    learnFrom(el.textContent);
    console.log("[AntiSpoil] Masqué :", el.textContent.slice(0, 80));
}

function scan() {
    document.querySelectorAll(
        "p, li, article, h1, h2, h3, h4"
    ).forEach(el => {
        if (!el.textContent) return;

        const txt = el.textContent.trim();

        // contenu trop court = bruit
        if (txt.length < 60) return;

        // éviter de masquer un conteneur géant
        if (el.scrollHeight > window.innerHeight * 1.5) return;

        if (el.closest(".anti-spoil-mask")) return;
        if (STATE.maskTitlesOnly && !isTitle(el)) return;

        if (STATE.hideComments && isLikelyCommentNode(el)) return;
        if (HOST.includes("youtube.com")) {
            scanYouTube();
            return;
        }
        if (HOST.includes("reddit.com")) {
            scanReddit();
            return;
        }

        if (shouldMask(txt)) mask(el);
    });
}


function init() {
    scan();

    const observer = new MutationObserver(() => {
        // scan simple (Toolix style) : fiable avant d'être “smart”
        scan();
    });

    observer.observe(document.body, { childList: true, subtree: true });
}
